/*
 * QrfeReaderInterface.h
 *
 *  Created on: 15.01.2009
 *      Author: stefan.detter
 */

#ifndef QRFEPREADERINTERFACE_H_
#define QRFEPREADERINTERFACE_H_

#include <QObject>
#include <QIODevice>
#include <QStringList>
#include <QMutex>
#include <QThread>
#include <QTimer>
#include <QTime>
#include <QMap>
#include <QPair>

#include <QrfeTraceModule>

#include "QrfeReaderInterfaceGlobal.h"

class QrfeProtocolHandler;
class QrfeTagManipulatorInterface;

class QrfeReaderInterface : public QObject, protected QrfeTraceModule
{
	Q_OBJECT

public:
	QrfeReaderInterface(QrfeProtocolHandler* ph, QObject* parent = 0);
	virtual ~QrfeReaderInterface();

public:
	virtual QrfeGlobal::Result initDevice( ) = 0;
	virtual QrfeGlobal::Result readSettings( ) = 0;

	virtual QrfeGlobal::Result setReaderName ( const QString &name ) = 0;
	virtual QrfeGlobal::Result setReaderLocation ( const QString &location ) = 0;

	virtual QrfeGlobal::Result getMaxAttenuation ( ushort &maxAttenuation ) = 0;
	virtual QrfeGlobal::Result getCurrentAttenuation ( ushort &currentAttenuation ) = 0;

	virtual QrfeGlobal::Result getMaxSensitivity ( short &maxSensitivity ) = 0;
	virtual QrfeGlobal::Result getMinSensitivity ( short &minSensitivity ) = 0;
	virtual QrfeGlobal::Result getCurrentSensitivity ( short &currentSensitivity ) = 0;

	virtual QrfeGlobal::Result getFrequency ( uchar &mode, uchar &maxFrequencyCount, QList<uint> &frequencies ) = 0;

	virtual QrfeGlobal::Result reboot ( ) = 0;
	virtual QrfeGlobal::Result setHeartBeat ( bool on, ushort interval = 0 ) = 0;
	virtual QrfeGlobal::Result setAntennaPower ( bool on ) = 0;
	virtual QrfeGlobal::Result setAttenuation ( ushort value ) = 0;
	virtual QrfeGlobal::Result setSensitivity ( short targetValue, short &actualValue ) = 0;
	virtual	QrfeGlobal::Result switchSystem ( ) = 0;

	virtual QrfeGlobal::Result setFrequency ( uchar mode, QList<uint> frequencies ) = 0;

	virtual QrfeGlobal::Result getParam ( ushort address, QByteArray &value ) = 0;
	virtual QrfeGlobal::Result setParam ( ushort address, QByteArray value ) = 0;

	virtual QrfeGlobal::Result setAntennaSequence ( const QList<QPair<uchar,ulong> > & sequence ) = 0;
	virtual QrfeGlobal::Result getAntennaSequence ( QList<QPair<uchar,ulong> > & sequence ) = 0;
	virtual QrfeGlobal::Result setWorkingAntenna ( uchar antennaIndex) = 0;
	virtual QrfeGlobal::Result getWorkingAntenna ( uchar & antennaIndex) = 0;

	virtual QrfeGlobal::Result setTagType(QrfeGlobal::TagType type) = 0;

	virtual QrfeGlobal::Result doSingleInventory ( QList<QrfeGlobal::TagEvent> &epc ) = 0;

	virtual QrfeGlobal::Result startCyclicInventory ( ulong time = 0 ) = 0;
	virtual QrfeGlobal::Result stopCyclicInventory ( ) = 0;

	virtual QrfeTagManipulatorInterface* getTagManipulator();

	virtual void 	getRSSIChannels(QList<QString>& names, QList<long>& minimum, QList<long>& maximum, QList<QString>& units) = 0;
	virtual QString getNotificationName(uchar id) = 0;
	virtual void 	getApplicationInformation( const QByteArray& data, QMap<uint,QPair<QString,QString> >& genericInfo) = 0;

public:
	QString 					name() const;
	QString 					readerId() const;

	QString 					serialNumber() const;
	ulong 						serialNumberExact() const;

	QrfeGlobal::ReaderType		readerType() const;
	ulong 						readerTypeExact() const;
	QString						readerTypeExactString() const;

	QrfeGlobal::ReaderModule	readerModule() const;
	QString						readerModuleString() const;

	QrfeGlobal::TagType			tagType() const;
	QString 					tagTypeString() const;

	ulong 						hardwareRevision() const;
	QString 					hardwareRevisionString() const;
	ulong 						softwareRevision() const;
	QString						softwareRevisionString() const;
	ulong 						bootloaderRevision() const;
	QString						bootloaderRevisionString() const;

	QString 					buildDate() const;
	QString 					buildTime() const;
	QString 					buildRevision() const;

	QrfeGlobal::HandlerState 	currentState() const;
	QString 					currentStateString() const;

	QrfeGlobal::HandlerAction 	currentAction() const;
	QString 					currentActionString() const;

	QrfeGlobal::ReaderSystem	currentSystem() const;
	QString						currentSystemString() const;

	qulonglong					statusRegister() const;
	QStringList					statusRegisterBitDescriptions() const;

	ulong 						gpioValues() const;

	uchar						antennaCount() const;

	QString						deviceName() const;
	QrfeGlobal::DeviceType		deviceType() const;

	ulong  						readerCaps() const;
	ulong  						tagFunctionCaps() const;

	QString 					readerName() const;
	QString 					readerLocation() const;

protected:
	virtual QStringList statusStrings() const;

	void 	setState(QrfeGlobal::HandlerState state);
	void 	setAction(QrfeGlobal::HandlerAction action);

	void  	handleProtocolHandlerError(const QrfeGlobal::Result res);

protected slots:
	void   handleProtocolHandlerState(int state);

public slots:
	void heartBeatTimeOut();

	virtual void deviceWasRemoved();

	void destroy();

signals:
	void heartBeat();

    void signatureChanged();

	void tagTypeChanged(int type);

	void changedState(int state);
	void changedAction(int action);
	void changedStatusRegister(qulonglong statusRegister);
	void changedGpioValues(ulong gpioValues);

	void changedRssiEnable(bool on, uchar rssiChildCount, const QStringList& rssiChildNames, int rssiChildMin, int rssiChildMax);
	void changedReadFrequencyEnable(bool on);

	void changedReaderName(const QString& name);
	void changedReaderLocation(const QString& location);

	void requestForDelete();
    void lostConnection();

	void inventoryStarted();
	void inventoryStoped();

	void cyclicInventory(const QrfeGlobal::TagReadEvent& tag);
	void exclusive_cyclicInventory(const QrfeGlobal::TagReadEvent& tag);

	void notification ( uchar id, const QVariant& value );

signals:
	void resetHeartBeat();

protected:

	QrfeProtocolHandler*		m_ph;

	QString 					m_readerName;
	QString						m_readerLocation;
	QString 					m_readerID;

	QString 					m_serialNumber;
	ulong 						m_serialNumberExact;

	QrfeGlobal::ReaderType		m_readerType;
	ulong 						m_readerTypeExact;
	QString						m_readerTypeExactString;
	QrfeGlobal::ReaderModule	m_readerModule;
	QrfeGlobal::TagType			m_tagType;

	ulong	 					m_hardwareRevision;
	QString 					m_hardwareRevisionString;
	ulong						m_softwareRevision;
	QString						m_softwareRevisionString;
	ulong						m_bootloaderRevision;
	QString						m_bootloaderRevisionString;
	QrfeGlobal::ReaderSystem	m_currentSystem;
	QString						m_currentSystemString;

	QString 					m_buildDate;
	QString 					m_buildTime;
	QString 					m_buildRevision;

	QrfeGlobal::HandlerState 	m_currentState;
	QrfeGlobal::HandlerAction 	m_currentAction;
	qulonglong					m_statusRegister;
	ulong 						m_gpioValues;
	QMap<uchar,QString>			m_statusRegisterBitDescritption;
	uchar 						m_antennaCount;

	QTimer*						m_heartBeatControl;
	bool						m_heartBeatHappened;
	QTime						m_lastHeartBeat;
	ulong 						m_heartBeatInterval;
	bool 						m_heartBeatDuplex;

	bool						m_deviceInit;

	bool						m_deviceRemovalDetected;

public:
	static ulong 				_heartBeatMultiplier;
	static uint 				_traceLevel;
};


#endif /* QRFEPREADERINTERFACE_H_ */
